<?php
namespace Opencart\Catalog\Controller\Extension\Seopro\Startup;

class SeoUrl extends \Opencart\System\Engine\Controller {
	private array $data = [];

	public function index(): void {
		if ($this->config->get('config_seo_url') && $this->config->get('other_seopro_status')) {

			$store_id = $this->config->get('config_store_id');
			
			if ($store_id == 0) {
			    $store_url = $this->config->get('site_url');
			} else {
			    $this->load->model('setting/store');
			    $store_info = $this->model_setting_store->getStore($store_id);
			    $store_url = $store_info ? $store_info['url'] : $this->config->get('site_url');
			}

			$this->registry->set('url', new \Opencart\System\Library\Url($store_url));
			$this->url->addRewrite($this);

			$this->load->model('design/seo_url');

			// Decode URL
			if (isset($this->request->get['_route_'])) {
				$route = $this->request->get['_route_'];
				unset($this->request->get['_route_']);

				$parts = array_filter(explode('/', $route));
				list($last_part) = explode('.', array_pop($parts));
				array_push($parts, $last_part);

				foreach ($parts as $key => $value) {
					$seo_url_info = $this->model_design_seo_url->getSeoUrlByKeyword($value);

					if ($seo_url_info) {
						$this->request->get[$seo_url_info['key']] = html_entity_decode($seo_url_info['value'], ENT_QUOTES, 'UTF-8');
						
						unset($parts[$key]);
					}
				}

				// Fix URL
				if (isset($this->request->get['product_id'])) {
					$this->request->get['route'] = 'product/product';
					$this->request->get['path'] = $this->getCategoryPath($this->getProductMainCategory($this->request->get['product_id']));
					if (isset($this->request->get['manufacturer_id'])) {
						unset($this->request->get['manufacturer_id']);
					}
				} elseif (isset($this->request->get['path'])) {
					$this->request->get['route'] = 'product/category';
					$categories = explode('_', $this->request->get['path']);
					$category_id = array_pop($categories);
					$this->request->get['path'] = $this->getCategoryPath($category_id);
				} elseif (isset($this->request->get['manufacturer_id'])) {
					$this->request->get['route'] = 'product/manufacturer.info';
				} elseif (isset($this->request->get['information_id'])) {
					$this->request->get['route'] = 'information/information';
				} elseif (isset($this->request->get['article_id'])) {
					$this->request->get['route'] = 'cms/blog.info';
					$topic_id = $this->getArticleTopic($this->request->get['article_id']);
					if ($topic_id) {
						$this->request->get['topic_id'] = $topic_id;
					}
				} elseif (isset($this->request->get['topic_id']) && !isset($this->request->get['article_id'])) {
					$this->request->get['route'] = 'cms/blog';
				}
				
				if (!isset($this->request->get['route'])) {
					$this->request->get['route'] = $this->config->get('action_default');
				}

				if ($parts) {
					$this->request->get['route'] = $this->config->get('action_error');
				}
			}

			$this->validate();
		}
	}

	public function rewrite(string $link): string {
		$url_info = parse_url(str_replace('&amp;', '&', $link));

		// Build the url
		$url = '';

		if ($url_info['scheme']) {
			$url .= $url_info['scheme'];
		}

		$url .= '://';

		if ($url_info['host']) {
			$url .= $url_info['host'];
		}

		if (isset($url_info['port'])) {
			$url .= ':' . $url_info['port'];
		}

		$base_url = $url;

		// Start changing the URL query into a path
		$paths = [];

		if (isset($url_info['query'])) {
			parse_str($url_info['query'], $query);

			// Parse the query into its separate parts
			$parts = explode('&', $url_info['query']);
		} else {
			$query = [];
			$parts = [];
		}

		$is_home = false;
		$is_product = false;
		$is_category = false;
		$is_manufacturer = false;
		$is_information = false;
		$is_topic = false;
		$is_article = false;

		switch ($query['route']) {
			case 'common/home':
				$is_home = true;
				break;
			case 'product/product':
				$is_product = true;
				break;
			case 'product/category':
				$is_category = true;
				break;
			case 'product/manufacturer.info':
				$is_manufacturer = true;
				break;
			case 'information/information':
				$is_information = true;
				break;
			case 'cms/blog':
				if (isset($url_info['query']) && str_contains($url_info['query'], 'topic_id=')) {
					$is_topic = true;
				}
				break;
			case 'cms/blog.info':
				$is_article = true;
				break;
			case 'information/information.info':
				return $link;
				break;
			default:
				break;
		}
		
		$language_id = $this->config->get('config_language_id');
		
		foreach ($parts as $part) {
			$pair = explode('=', $part);

			if (isset($pair[0])) {
				$key = (string)$pair[0];
			}

			if (isset($pair[1])) {
				$value = (string)$pair[1];
			} else {
				$value = '';
			}

			$index = $key . '=' . $value;

			if ($is_product) { 
				if ($key == 'path' || $key == 'manufacturer_id') {
					unset($query[$key]);
					continue;
				} elseif ($key == 'product_id') {
					if ($this->config->get('other_seopro_product_include') == 'category') {
						$category_paths = $this->model_design_seo_url->getSeoUrlByKeyValue('path', (string)$this->getCategoryPath($this->getProductMainCategory($value)));
						if ($this->config->get('other_seopro_category_flat') && !empty($category_paths['keyword'])) {
							$categories = explode('/', $category_paths['keyword']);
							$category_paths['keyword'] = array_pop($categories);
						}
						$paths[] = $category_paths;
					} elseif ($this->config->get('other_seopro_product_include') == 'manufacturer') {
						$paths[] = $this->model_design_seo_url->getSeoUrlByKeyValue('manufacturer_id', (string)$this->getProductManufacturer($value));
					}
				}
			} elseif ($is_article) {
				if ($key == 'topic_id') {
					unset($query[$key]);
					continue;
				} elseif ($key == 'article_id') {
					$paths[] = $this->model_design_seo_url->getSeoUrlByKeyValue('topic_id', (string)$this->getArticleTopic($value));
				}
			}
			
			if (!isset($this->data[$language_id][$index])) {
				$this->data[$language_id][$index] = $this->model_design_seo_url->getSeoUrlByKeyValue((string)$key, (string)$value);
			}
			
			$result = [];
			
			if ($this->data[$language_id][$index]) {
				if ($key == 'path' && $this->config->get('other_seopro_category_flat')) {
					$categories = explode('/', $this->data[$language_id][$index]['keyword']);
					$this->data[$language_id][$index]['keyword'] = array_pop($categories);
				}
				$result = [
					'keyword' => $this->data[$language_id][$index]['keyword'],
					'key' => $key,
					'value' => $value,
					'sort_order' => $this->data[$language_id][$index]['sort_order'],
				];
			}

			if (!empty($result['keyword'])) {
				if ($is_product && $key == 'product_id') {
					$result['keyword'] .= $this->config->get('other_seopro_product_end');
					$result['sort_order'] = 999;
				} elseif ($is_category && $key == 'path') {
					$result['keyword'] .= $this->config->get('other_seopro_category_end');
					$result['sort_order'] = 999;
				} elseif ($is_manufacturer && $key == 'manufacturer_id') {
					$result['keyword'] .= $this->config->get('other_seopro_manufacturer_end');
					$result['sort_order'] = 999;
				} elseif ($is_information && $key == 'information_id') {
					$result['keyword'] .= $this->config->get('other_seopro_information_end');
					$result['sort_order'] = 999;
				} elseif ($is_topic && $key == 'topic_id') {
					$result['keyword'] .= $this->config->get('other_seopro_topic_end');
					$result['sort_order'] = 999;
				} elseif ($is_article && $key == 'article_id') {
					$result['keyword'] .= $this->config->get('other_seopro_article_end');
					$result['sort_order'] = 999;
				} elseif ($key == 'language') {
					$result['sort_order'] = -999;
				}
			}

			if ($result) {
				$paths[] = $result;
				unset($query[$key]);
			}
		}

		$paths = array_filter($paths);

		// Sort
		$sort_order = [];
		foreach ($paths as $key => $value) {
			$sort_order[$key] = $value['sort_order'];
		}
		array_multisort($sort_order, SORT_ASC, $paths);

		// Build the path
		$url .= str_replace('/index.php', '', $url_info['path'] ?? '');

		// Skip paths
		$skip = [
			['route' => 'common/home']
		];

		if ($this->config->get('other_seopro_language_dir') || $is_home) {
			$skip[] = ['language' => $this->config->get('other_seopro_language_default')];
		}
		if ($this->config->get('other_seopro_product_dir')) {
			$skip[] = ['route' => 'product/product'];
		}
		if ($this->config->get('other_seopro_category_dir')) {
			$skip[] = ['route' => 'product/category'];
		}
		if ($this->config->get('other_seopro_manufacturer_dir')) {
			$skip[] = ['route' => 'product/manufacturer.info'];
		}
		if ($this->config->get('other_seopro_information_dir')) {
			$skip[] = ['route' => 'information/information'];
		}
		if ($this->config->get('other_seopro_blog_dir')) {
			$skip[] = ['route' => 'cms/blog.info'];
			if ($is_topic) {
				$skip[] = ['route' => 'cms/blog'];
			}
		}

		foreach ($paths as $result) {
			if (!empty($result['key']) && !empty($result['value'])) {
				if (!in_array([$result['key'] => $result['value']], $skip, true)) {
					$url .= '/' . $result['keyword'];
				} elseif ($result['value'] === 'common/home') {
					$url .= '/';
				}
			}
		}

		// Rebuild the URL query
		if ($url == $base_url) {
			$url .= '/index.php';
		}

		if ($query) {
			$url .= '?' . str_replace(['%2F'], ['/'], http_build_query($query));
		}

		return $url;
	}

	private function validate(): void {
		$uri = ltrim($this->request->server['REQUEST_URI'], '/');

		if (!isset($this->request->get['route']) && substr($uri, 0, 9) === 'index.php') {
			$this->response->redirect($this->config->get('config_url'), '301');
			return;
		}
		
		$route = $this->request->get['route'] ?? 'error/not_found';
		
		if ($route == 'error/not_found' || php_sapi_name() === 'cli') {
			return;
		}

		if (isset($this->request->server['HTTP_X_REQUESTED_WITH']) && strtolower($this->request->server['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
			$this->response->addHeader('X-Robots-Tag: noindex');
			return;
		}

		if ($uri === 'sitemap.xml' && $this->config->get('feed_sitemap_ml_status')) {
			$this->request->get['route'] = 'extension/sitemap_ml/feed/sitemap_ml'; 
			return;
		}

		$host = substr($this->config->get('config_url'), 0, $this->strposOffset('/', $this->config->get('config_url'), 3) + 1);
		
		$url = str_replace('&amp;', '&', $host . $uri);

		$queryString = $this->getQueryString(['route']);
		
		if (!$this->config->get('other_seopro_language_dir') && empty($this->request->get['language'])) {
			$seo = $this->url->link($route, $queryString . '&language=' . $this->config->get('other_seopro_language_default'));
		} else {
			$seo = $this->url->link($route, $queryString);
		}
		
		$seo = str_replace('&amp;', '&', $seo);

		if (rawurldecode($url) !== rawurldecode($seo)) {
			$this->response->redirect($seo, '301');
		}
	}

	private function getQueryString($exclude = []) {
		$filteredQuery = array_diff_key($this->request->get, array_flip($exclude));
		return urldecode(http_build_query($filteredQuery));
	}

	private function strposOffset($needle, $haystack, $entry) {
		if ($entry <= 0) {
			return false;
		}

		$position = 0;
		for ($i = 1; $i <= $entry; $i++) {
			$position = strpos($haystack, $needle, $position);
			if ($position === false) {
				return false;
			}
			$position++;
		}

		return $position - 1;
	}

	private function getCategoryPath($category_id) {
		$language_id = $this->config->get('config_language_id');
		if (isset($this->data[$language_id]['category_paths'][$category_id])) {
			return $this->data[$language_id]['category_paths'][$category_id];
		}

		$path = [];
		while ($category_id) {
			$query = $this->db->query("SELECT parent_id, category_id FROM " . DB_PREFIX . "category WHERE category_id = '" . (int)$category_id . "'");
			if ($query->num_rows) {
				$path[] = $query->row['category_id'];
				$category_id = $query->row['parent_id'];
			} else {
				break;
			}
		}

		$full_path = implode('_', array_reverse($path));
		$this->data[$language_id]['category_paths'][$category_id] = $full_path;

		return $full_path;
	}

	private function getProductMainCategory($product_id) {
		$query = $this->db->query("SELECT category_id FROM " . DB_PREFIX . "product_to_category WHERE product_id = '" . (int)$product_id . "' ORDER BY main_category DESC LIMIT 1");
		
		if (isset($query->row['category_id'])) {
			return $query->row['category_id'];
		} else {
			return 0;
		}
	}

	private function getProductManufacturer($product_id) {
		$query = $this->db->query("SELECT manufacturer_id FROM " . DB_PREFIX . "product WHERE product_id = '" . (int)$product_id . "' LIMIT 1");
		
		if (isset($query->row['manufacturer_id'])) {
			return $query->row['manufacturer_id'];
		} else {
			return 0;
		}
	}

	private function getArticleTopic($article_id) {
		$query = $this->db->query("SELECT topic_id FROM `" . DB_PREFIX . "article` WHERE `article_id` = '" . (int)$article_id . "'");
		
		if (isset($query->row['topic_id'])) {
			return $query->row['topic_id'];
		} else {
			return 0;
		}
	}
}